/*---------------------------------------------------------------------------*\

	FILE....: TRAIN.CPP
	TYPE....: C++ Module
	AUTHOR..: David Rowe
	DATE....: 18/8/98

	Functions for training VPB programmable tones.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*\

	INCLUDES

\*---------------------------------------------------------------------------*/

#include "train.h"
#include "../../src/vpbapi.h"
#include "../../src/wavecpp.h"
#include "../../src/wobbly.h"

#include <assert.h>
#include <math.h>
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>



/*---------------------------------------------------------------------------*\

DEFINES and TYPEDEFS

\*---------------------------------------------------------------------------*/

enum States {LOOK_FOR_ON, LOOK_FOR_OFF};

#define	N	80		// processing block size
#define	NFFT	(2*TRAIN_NSPEC)	// FFT size (=512)
#define L2NFFT  9               // base 2 log of NFFT
#define	NFFT2	TRAIN_NSPEC	// useful constant
#define	FS	8000		// sample rate
#define	WIN_OFFSET	1000	// offset to get FFT window in good signal area
#define	EBUF_MAX	300	// maximum of 3 seconds of energy samples
#define	AMP_MAX		32767	// maximum amplitude
#define	FMIN		300	// minimum frequency
#define	FMAX		3300	// maximum frequency

/*
#define DEBUG
#define DEBUGFAKE
*/

/* Now using defN from fft.h
typedef struct {
	float	real,imag;
} COMP;
*/

/*---------------------------------------------------------------------------*\

	STATICS

\*---------------------------------------------------------------------------*/

// prototype continous tone

static VPB_DETECT continuous = {
	2,		// number of cadence states
	1000,		// tone id, caller must fill in
	0,		// number of tones
	0,		// freq1
	0,		// bandwidth1
	0,		// freq2
	0,		// bandwidth2
	-40,		// level1
	-40,		// level2
	10,		// twist
	10,		// SNR
	40,		// glitchs of 40ms ignored

	{
		{
			VPB_RISING,	// state 0
			0,
			0,
			0,
		},

		{
			VPB_TIMER,	// state 1
			2000,
			0,
			0
		}
	}
};

// prototype cadenced tone

static VPB_DETECT cadenced = {
	3,		// number of cadence states
	1000,		// tone id, filled in by user
	0,		// number of tones
	0,		// freq1
	0,		// bandwidth1
	0,		// freq2
	0,		// bandwidth2
	-20,		// level1
	-20,		// level2
	10,		// twist
	10,		// SNR
	40,		// glitchs of 40ms ignored

	{
		{
			VPB_RISING,	// state 0			
			0,
			0,
			0
		},

		{
			VPB_FALLING,	// state 1			
			0,
			0,		// ton min		
			0		// ton max		
		},

		{
			VPB_RISING,	// state 1			
			0,
			0,		// ton min		
			0		// ton max		
		}
	}
};

/*---------------------------------------------------------------------------*\

	FUNCTION HEADERS

\*---------------------------------------------------------------------------*/

int determine_frequency(TRAIN *tone, short buf[], int nsam);
int determine_cadence(TRAIN *tone, short buf[], int nsam);

/*---------------------------------------------------------------------------*\

	FUNCTIONS

\*---------------------------------------------------------------------------*/


/*--------------------------------------------------------------------------*\

	FUNCTION.: train_train()
	AUTHOR...: David Rowe
	DATE.....: 20/8/98

	Trains a VPB tone detector given a buffer of samples.  The caller must
	enumerate the tone.

	Returns TRAIN_OK if OK, else negative error code.

\*--------------------------------------------------------------------------*/

int WINAPI train_train(
	VPB_DETECT 	*tone,		// programmable tone definition
	TRAIN		*train,		// params controlling algorithm
	short		buf[],		// input samples (16 bit linear, 8 kHz)
	int		nsam		// number of input samples
)
{
	// validate 
	assert(tone != NULL);
	assert(train != NULL);
	assert(buf != NULL);

	if (nsam <= 0)
		return(TRAIN_NSAM_TO_SMALL);
	if (train->OnThresh > 0)
		return (TRAIN_INVALID_ON_THRESH);
	if (train->OffThresh > 0)
		return (TRAIN_INVALID_OFF_THRESH);
	if (train->CadenceWindow <= 0)
		return(TRAIN_INVALID_CADENCE_WINDOW);
	if (train->Bandwidth <= 0)
		return(TRAIN_INVALID_BANDWIDTH);
	if ((train->eThresh <=0) || (train->eThresh > 1.0))
		return(TRAIN_INVALID_E_THRESH);

	// analyse cadence and frequency
	int ret;
	ret = determine_cadence(train, buf, nsam);
	if (ret != TRAIN_OK)
		return ret;
	#ifdef DEBUG
	printf("nCadMark: %d \n",train->nCadMark);
	#endif
	// We now want to find out the freqencies for each cadence
	// But we seem to be only performing a single frequency
	// determination
	if (train->nCadMark) {
		ret = determine_frequency(train, buf, nsam);
		if (ret != TRAIN_OK)
			return ret;
	}

	// construct programmable tone using extraced parameters ------------
	if (train->nCadMark < 1){
		return TRAIN_NOT_ENOUGH_CADENCE;
	}
	else if (train->nCadMark < 2) {
		memcpy(tone, &continuous, sizeof(VPB_DETECT));
		tone->ntones = train->nSpecMark;

		// setup frequencies and bandwidths
		tone->freq1 = train->SpectrumM[0];
		tone->bandwidth1 = train->Bandwidth;
		tone->minlevel1 = train->SpecMag[0]-train->MagTol;
		if (tone->ntones == 2) {
			tone->freq2 = train->SpectrumM[1];
			tone->bandwidth2 = train->Bandwidth;
			tone->minlevel2 = train->SpecMag[1]-train->MagTol;
		}

	} 
	else {
		memcpy(tone, &cadenced, sizeof(VPB_DETECT));
		tone->ntones = train->nSpecMark;

		// setup frequencies and bandwidths
		tone->freq1 = train->SpectrumM[0];
		tone->bandwidth1 = train->Bandwidth;
		tone->minlevel1 = train->SpecMag[0]-train->MagTol;
		if (tone->ntones == 2) {
			tone->freq2 = train->SpectrumM[1];
			tone->bandwidth2 = train->Bandwidth;
			tone->minlevel2 = train->SpecMag[1]-train->MagTol;
		}

		// now set up candences
		/* first rising has no time values, it just kicks off 
		 * the state machine we time the second (falling) and 
		 * third (rising) states 
		 */

		int falling = train->CadenceM[1]-train->CadenceM[0];
		int rising = train->CadenceM[2]-train->CadenceM[1];

		tone->stran[1].tmin = falling - train->CadenceWindow/2;
		tone->stran[1].tmax = falling + train->CadenceWindow/2;
		tone->stran[2].tmin = rising - train->CadenceWindow/2;
		tone->stran[2].tmax = rising + train->CadenceWindow/2;
	}

	return TRAIN_OK;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: train_train_from_linear_wave()
	AUTHOR...: David Rowe
	DATE.....: 29/9/98

	Trains a VPB tone detector given a wave file containing linear samples.
	The caller must	enumerate the tone.

	Returns TRAIN_OK if OK, else negative error code.

\*--------------------------------------------------------------------------*/

int WINAPI train_train_from_linear_wave(
	VPB_DETECT 	*tone,		// programmable tone definition
	TRAIN		*train,		// params controlling algorithm
	char		file[]		// wave filename
)
{
	void		*wave;
	unsigned long	sizebytes;
	unsigned long	sizeshorts;
	unsigned short	mode;
	int		ret;

	try {

		// open wave file and determine size
		#ifdef DEBUG
		printf("Opening file %s \n",file);
		#endif
		wave_open_read(&wave, file);
		wave_get_size(wave, &sizebytes);
		sizebytes=(2*8000*5);
		sizeshorts = sizebytes/sizeof(short);
		#ifdef DEBUG
		printf("Size shorts %ld \n",sizeshorts);
		#endif

		// check if it is a linear file

		wave_get_mode(wave, &mode);
		if (mode != VPB_LINEAR) {
			vpb_wave_close_read(wave);
			return(TRAIN_WAVE_FILE_NOT_LINEAR);
		}

		// allocate a buffer, read samples, and train

		short *buf = new short[sizeshorts];

		if (buf == NULL) {
			vpb_wave_close_read(wave);
			return(TRAIN_CANT_ALLOC_BUF);
		}

		ret = vpb_wave_read(wave, (char*)buf, sizebytes);
		#ifdef DEBUG
		printf("Read %d bytes from file\n",ret);
		printf("Was trying to get %ld bytes\n",sizebytes);
		#endif
		if (ret != (int)sizebytes) {
			vpb_wave_close_read(wave);
			delete [] buf;
			return(TRAIN_WAVE_READ_ERROR);
		}

		ret = train_train(tone,	train, buf, sizeshorts);

		// clean up and exit
		delete [] buf;
		vpb_wave_close_read(wave);
	}

	catch (Wobbly w) {
//		w;
		return(TRAIN_WAVE_FILE_ERROR);
	}

	return(ret);		
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: determine_cadence()
	AUTHOR...: David Rowe
	DATE.....: 20/8/98

	Determines the cadence of a tone contained in a buffer of speech
	samples.

\*--------------------------------------------------------------------------*/

int determine_cadence(
	TRAIN *train, 
	short buf[], 
	int nsam
)
{
	int	i,j,k;
	float	x;

//	FILE *f = fopen("/xhome1/tmp/energy.txt","wt");
//	assert (f!=NULL);


	// first determine energy of each N sample frame

	if ( (nsam/N) < TRAIN_NCAD){
		train->ncad = nsam/N;
	}
	else {
		train->ncad =  TRAIN_NCAD;
	}

	assert(train->ncad <= TRAIN_NCAD);

	if (train->ncad < 1)
		return(TRAIN_NSAM_TO_SMALL);

	float energy[TRAIN_NCAD];

	for(i=0,j=0; i<nsam; i+=N,j++) {
		energy[j] = (float)0.0;
		for(k=0; k<N; k++) {
			x = buf[i+k];
			energy[j] += pow(x, (float)2.0); 
		}

		// convert to dB ref 0dB as overload of single sinewave
		energy[j] = (float)(10.0*log10((energy[j]+1.0)/N) - 87.3);
		train->Cadence[j] = (short)energy[j];
		train->CadenceX[j] = j*(N*1000)/FS;
//		fprintf(f,"%d\t%f\n",train->CadenceX[j],energy[j]);
	}
//	fclose(f);

	// Now look for large energy differences between samples,
	// which indicate transitions from on-off or off-on cadence state
	// start by looking for on transitions
	States state = LOOK_FOR_ON;
	States nextState;

	int marks = 0;
	j = 1;
	int t = 0;
	do {

		nextState = state;

		switch(state) {
			case LOOK_FOR_ON:
			if (energy[j] > train->OnThresh) {
				nextState = LOOK_FOR_OFF;
				train->CadenceM[marks++] = t;
			}

			break;

			case LOOK_FOR_OFF:
			if (energy[j] < train->OffThresh) {
				nextState = LOOK_FOR_ON;
				train->CadenceM[marks++] = t;
			}
			break;

			default:
			assert(0);
		}

		state = nextState;
		j++;
		t += N*1000/FS;

	} while((j < (train->ncad-1)) && (marks < TRAIN_NMARK));

	train->nCadMark = marks;

	return(TRAIN_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: determine_frequency()
	AUTHOR...: David Rowe
	DATE.....: 20/8/98

	Determines the frequency of a tone or tone pair contained in a buffer
	of speech samples.

\*--------------------------------------------------------------------------*/

int determine_frequency(
	TRAIN *train, 
	short buf[], 
	int nsam
)
{
	int	i,j;
	COMPLX	x[NFFT];
	float	two_pi = (float)(2.0*acos(-1.0));
	float	h;
	int	mark = 0;
	float	eline;
	float	r = (float)FS/NFFT; train->r = r;
	float	Ewin,Etd;
	int	bmin = (int)floor(FMIN/r+0.5);
	int	bmax = (int)floor(FMAX/r+0.5);

	// extract samples, window, and FFT

	// start is the time of the first cadence * Sample rate
	//  divided by 1000 window offset 
	//  (window offset = to get FFT window in good signal area)
	//  ie the sample to start checking at... lets take 100ms
	//  100*8000/2000=400th sample
	int start = train->CadenceM[0]*FS/1000+WIN_OFFSET;
	Ewin = (float)0.0;
	Etd = (float)0.0;
	
	// Put mangled buf info into Complx array to pass to the FDFT
	// We copy NFFT samples to the Complx array (512)
	for(i=0,j=start; i<NFFT; i++,j++) {
		h = (float)(0.5 - 0.5*cos(two_pi*i/NFFT));
		Ewin += h*h;
		x[i].real = (float)(buf[j]*h);

		#ifdef DEBUGFAKE
		x[i].real = 0;
		x[i].real = cos(two_pi*i*0.125);
		#endif

		x[i].imag = (float)0.0;
		Etd += x[i].real*x[i].real;
	}
	//x[0].real = 1;
	
	// Call the FFT routine

	#ifdef DEBUG
	printf("before fft....\n");
	for (i=0; i<NFFT; i++){
		printf("sample %d real: %f imag: %f \n",i,x[i].real,x[i].imag);
	}
	#endif
	dft(x,NFFT);

	
	#ifdef DEBUG
	printf("after fft....\n");
	for (i=0; i< NFFT2; i++){
		printf("\t%d\t%f\t%f \n",i,x[i].real,x[i].imag);
	}
	#endif

	#ifdef DEBUGFAKE
	exit(0);
	#endif

	// determine energy of each + freq bin
	float e[NFFT2];
	float totale = (float)0.0;
	for(i=0; i<NFFT2; i++) {
		e[i] = x[i].real*x[i].real + x[i].imag*x[i].imag;
		train->Spectrum[i] = (short)(10.0*log10(e[i]+1.0));
		train->SpectrumX[i] = (short)(i*r);
		totale += e[i];
	}

	// search out the single or dual peaks

	do {

		// Determine global maxima, this is the centre of the freuqency 
		float max = (float)0.0;
		int maxpos = 0;
		for(i=bmin; i<bmax; i++) {
			if (e[i] > max) {
				maxpos = i;
				max = e[i];
			}
		}

		// determine vital statistics of frequency
		float maxf = maxpos*r;
		int start_bin = (int)((maxf - train->Bandwidth/2)/r);
		int end_bin = (int)((maxf + train->Bandwidth/2)/r);

		// check limits of start and end of frequency component
		if (start_bin < 0) {
			start_bin = 0;
			end_bin = (int)((start_bin + train->Bandwidth)/r);
		}
		if (end_bin >= NFFT2) {
			end_bin = NFFT2-1;
			start_bin = (int)((end_bin - train->Bandwidth)/r);
		}

		// Sum total energy in bandwidth about maxima
		// Then zero out this band for next iteration
		eline = (float)1.0;
		for(i=start_bin; i<=end_bin; i++) {
			eline += e[i];
			e[i] = (float)0.0;
		}
		train->SpectrumM[mark] = (short)maxf;

		// determine line magnitude
		float mag = (float)sqrt((4.0*eline)/(NFFT*Ewin));
		float magdB = (float)(20*log10((mag+1.0)/AMP_MAX));
		train->SpecMag[mark++] = (short)(magdB);

		/* If most of the spectrum energy is in the first tone,
		 * then exit the while loop.  Otherwise have another 
		 * iteration as significant energy still exists, which 
		 * indicates the presence of a second tone
		 */

	} while ((mark < 2) && ((eline/totale) < train->eThresh));

	train->nSpecMark = mark;

	return(TRAIN_OK);
}


/* Discrete Fourier Transform
 * Demonstration code by Emil Mikulic, 2001.
 * http://members.optushome.com.au/darkmoon7/sound/dft/
 *
 * Feel free to do whatever you like with this code.
 * Feel free to credit me.
 *
 * Modified by Ben Kramer to suit our application
 * Date: 10/02/2003
 */
void dft(COMPLX *data, int samples)
{
	int r, k;
	COMPLX t[samples/2+1];

	for (r=0; r<=samples/2; r++)
	{
		t[r].real = (float)0;
		t[r].imag = (float)0;
		for (k=0; k<samples; k++)
		{
			float theta = (float)(r*k) * -2.0 * M_PI /
				(float)samples;

			t[r].real += data[k].real * cos(theta);
			t[r].imag += data[k].real * sin(theta);
		}
	}
	for (r=0; r<=samples/2; r++)
	{
			data[r].real = t[r].real;
			data[r].imag = t[r].imag;
	}
}
